/* ============================================
   The Lookeron School - Main JavaScript
   ============================================ */

// ============================================
// HTML Include Loader
// ============================================
async function loadIncludes() {
    // Detect if we're in a subfolder (like /pages/) and adjust path accordingly
    const currentPath = window.location.pathname;
    let basePath = '';
    
    // Check if we're in a subfolder
    if (currentPath.includes('/pages/') || currentPath.includes('/php/')) {
        basePath = '../';
    }
    
    // Load header
    const headerPlaceholder = document.getElementById('header-placeholder');
    if (headerPlaceholder) {
        try {
            const response = await fetch(basePath + 'includes/header.html');
            if (response.ok) {
                let html = await response.text();
                
                // If in subfolder, update all relative links in header
                if (basePath) {
                    html = html.replace(/href="(?!http|#|mailto|tel)([^"]+)"/g, 'href="' + basePath + '$1"');
                    html = html.replace(/src="(?!http|data:)([^"]+)"/g, 'src="' + basePath + '$1"');
                }
                
                headerPlaceholder.innerHTML = html;

                // Highlight active nav link based on current page
                highlightActiveNavLink();
            }
        } catch (error) {
            console.error('Error loading header:', error);
        }
    }

    // Load footer
    const footerPlaceholder = document.getElementById('footer-placeholder');
    if (footerPlaceholder) {
        try {
            const response = await fetch(basePath + 'includes/footer.html');
            if (response.ok) {
                let html = await response.text();
                
                // If in subfolder, update all relative links in footer
                if (basePath) {
                    html = html.replace(/href="(?!http|#|mailto|tel)([^"]+)"/g, 'href="' + basePath + '$1"');
                    html = html.replace(/src="(?!http|data:)([^"]+)"/g, 'src="' + basePath + '$1"');
                }
                
                footerPlaceholder.innerHTML = html;
            }
        } catch (error) {
            console.error('Error loading footer:', error);
        }
    }

    // Re-initialize after loading includes
    initializeAfterIncludes();
}

// Highlight active navigation link
function highlightActiveNavLink() {
    const currentPage = window.location.pathname.split('/').pop().replace('.html', '') || 'index';

    // Desktop nav links
    document.querySelectorAll('.nav-link').forEach(link => {
        const page = link.getAttribute('data-page');
        if (page === currentPage) {
            link.classList.add('text-primary', 'border-b-2', 'border-primary', 'pb-1');
            link.classList.remove('text-gray-700');
        }
    });

    // Mobile nav links
    document.querySelectorAll('.mobile-nav-link').forEach(link => {
        const page = link.getAttribute('data-page');
        if (page === currentPage) {
            link.classList.add('bg-primary', 'text-white');
            link.classList.remove('text-gray-700', 'hover:bg-gray-100');
        }
    });
}

// Initialize components after includes are loaded
function initializeAfterIncludes() {
    // Re-setup mobile menu
    setupMobileMenu();

    // Update current year
    document.querySelectorAll('.current-year').forEach(el => {
        el.textContent = new Date().getFullYear();
    });
}

// Setup mobile menu (called after includes load)
function setupMobileMenu() {
    const mobileMenuBtn = document.getElementById('mobile-menu-btn');
    const mobileMenu = document.getElementById('mobile-menu');
    const closeMenuBtn = document.getElementById('close-menu');
    const menuOverlay = document.getElementById('menu-overlay');

    function openMenu() {
        if (mobileMenu) {
            mobileMenu.classList.add('active');
        }
        if (menuOverlay) {
            menuOverlay.classList.remove('hidden');
        }
        document.body.style.overflow = 'hidden';
    }

    function closeMenu() {
        if (mobileMenu) {
            mobileMenu.classList.remove('active');
        }
        if (menuOverlay) {
            menuOverlay.classList.add('hidden');
        }
        document.body.style.overflow = '';
    }

    if (mobileMenuBtn) {
        mobileMenuBtn.addEventListener('click', openMenu);
    }

    if (closeMenuBtn) {
        closeMenuBtn.addEventListener('click', closeMenu);
    }

    if (menuOverlay) {
        menuOverlay.addEventListener('click', closeMenu);
    }
}

// Wait for DOM to be ready
document.addEventListener('DOMContentLoaded', function () {

    // Load includes first (if using include system)
    loadIncludes();

    // Initialize Lucide Icons
    if (typeof lucide !== 'undefined') {
        lucide.createIcons();
    }

    // Initialize AOS (Animate on Scroll)
    if (typeof AOS !== 'undefined') {
        AOS.init({
            duration: 800,
            once: true,
            offset: 100
        });
    }

    // ============================================
    // Mobile Menu Functionality
    // ============================================
    const mobileMenuBtn = document.getElementById('mobile-menu-btn');
    const mobileMenu = document.getElementById('mobile-menu');
    const closeMenuBtn = document.getElementById('close-menu');
    const menuOverlay = document.getElementById('menu-overlay');

    function openMenu() {
        if (mobileMenu) {
            mobileMenu.classList.add('active');
        }
        if (menuOverlay) {
            menuOverlay.classList.remove('hidden');
        }
        document.body.style.overflow = 'hidden';
    }

    function closeMenu() {
        if (mobileMenu) {
            mobileMenu.classList.remove('active');
        }
        if (menuOverlay) {
            menuOverlay.classList.add('hidden');
        }
        document.body.style.overflow = '';
    }

    if (mobileMenuBtn) {
        mobileMenuBtn.addEventListener('click', openMenu);
    }

    if (closeMenuBtn) {
        closeMenuBtn.addEventListener('click', closeMenu);
    }

    if (menuOverlay) {
        menuOverlay.addEventListener('click', closeMenu);
    }

    // Close menu on escape key
    document.addEventListener('keydown', function (e) {
        if (e.key === 'Escape' && mobileMenu && mobileMenu.classList.contains('active')) {
            closeMenu();
        }
    });

    // ============================================
    // Gallery Functionality
    // ============================================
    const galleryImages = [
        'images/gallery/a.webp',
        'images/gallery/b.webp',
        'images/gallery/c.webp',
        'images/gallery/d.webp',
        'images/gallery/e.webp',
        'images/gallery/f.webp',
        'images/gallery/g.webp',
        'images/gallery/h.webp',
        'images/gallery/i.webp',
        'images/gallery/j.webp',
        'images/gallery/k.webp',
        'images/gallery/1.webp',
        'images/gallery/2.webp',
        'images/gallery/3.webp',
        'images/gallery/4.webp',
        'images/gallery/5.webp',
        'images/gallery/6.webp',
        'images/gallery/7.webp',
        'images/gallery/8.webp',
        'images/gallery/9.webp',
        'images/gallery/10.webp',
        'images/gallery/11.webp',
        'images/gallery/12.webp',
        'images/gallery/13.webp',
        'images/gallery/14.webp',
        'images/gallery/15.webp',
        'images/gallery/16.webp',
        'images/gallery/17.webp',
        'images/gallery/18.webp',
        'images/gallery/19.webp',
        'images/gallery/20.webp',
        'images/gallery/21.webp',
        'images/gallery/22.webp',
        'images/gallery/23.webp',
        'images/gallery/24.webp',
        'images/gallery/25.webp',
        'images/gallery/26.webp',
        'images/gallery/27.webp',
        'images/gallery/28.webp',
        'images/gallery/29.webp',
        'images/gallery/31.webp',
        'images/gallery/32.webp',
        'images/gallery/33.webp',
        'images/gallery/34.webp',
        'images/gallery/35.webp',
        'images/gallery/36.webp',
        'images/gallery/37.webp',
        'images/gallery/38.webp',
        'images/gallery/39.webp',
        'images/gallery/40.webp',
        'images/gallery/41.webp',
        'images/gallery/42.webp',
        'images/gallery/43.webp',
        'images/gallery/44.webp',
        'images/gallery/45.webp',
        'images/gallery/46.webp'
    ];

    const galleryGrid = document.getElementById('gallery-grid');

    // Populate gallery if gallery grid exists
    if (galleryGrid) {
        galleryImages.forEach((src, index) => {
            const item = document.createElement('div');
            item.className = 'gallery-item';
            item.dataset.index = index;
            item.dataset.aos = 'fade-up';
            item.dataset.aosDelay = (index % 8) * 50;

            const img = document.createElement('img');
            img.src = src;
            img.alt = 'The Lookeron School Gallery Image ' + (index + 1);
            img.className = 'w-full h-48 md:h-56 object-cover';
            img.loading = 'lazy';

            // Create overlay with icon
            const overlay = document.createElement('div');
            overlay.className = 'gallery-overlay';
            overlay.innerHTML = '<i class="fa-solid fa-image"></i>';

            item.appendChild(img);
            item.appendChild(overlay);
            galleryGrid.appendChild(item);
        });

        // Re-initialize AOS after adding gallery items
        if (typeof AOS !== 'undefined') {
            AOS.refresh();
        }
    }

    // ============================================
    // Lightbox Functionality
    // ============================================
    const lightbox = document.getElementById('lightbox');
    const lightboxImg = document.getElementById('lightbox-img');
    const lightboxClose = document.getElementById('lightbox-close');
    const lightboxPrev = document.getElementById('lightbox-prev');
    const lightboxNext = document.getElementById('lightbox-next');
    let currentIndex = 0;

    // Add click listeners to gallery items
    if (galleryGrid) {
        document.querySelectorAll('.gallery-item').forEach((item, index) => {
            item.addEventListener('click', function () {
                currentIndex = parseInt(this.dataset.index) || index;
                if (lightboxImg) {
                    lightboxImg.src = galleryImages[currentIndex];
                }
                if (lightbox) {
                    lightbox.classList.add('active');
                }
                document.body.style.overflow = 'hidden';
            });
        });
    }

    // Lightbox close
    if (lightboxClose) {
        lightboxClose.addEventListener('click', function () {
            if (lightbox) {
                lightbox.classList.remove('active');
            }
            document.body.style.overflow = '';
        });
    }

    // Lightbox navigation
    if (lightboxPrev) {
        lightboxPrev.addEventListener('click', function (e) {
            e.stopPropagation();
            currentIndex = (currentIndex - 1 + galleryImages.length) % galleryImages.length;
            if (lightboxImg) {
                lightboxImg.src = galleryImages[currentIndex];
            }
        });
    }

    if (lightboxNext) {
        lightboxNext.addEventListener('click', function (e) {
            e.stopPropagation();
            currentIndex = (currentIndex + 1) % galleryImages.length;
            if (lightboxImg) {
                lightboxImg.src = galleryImages[currentIndex];
            }
        });
    }

    // Close lightbox on backdrop click
    if (lightbox) {
        lightbox.addEventListener('click', function (e) {
            if (e.target === lightbox) {
                lightbox.classList.remove('active');
                document.body.style.overflow = '';
            }
        });
    }

    // Keyboard navigation for lightbox
    document.addEventListener('keydown', function (e) {
        if (!lightbox || !lightbox.classList.contains('active')) return;

        if (e.key === 'Escape') {
            lightbox.classList.remove('active');
            document.body.style.overflow = '';
        } else if (e.key === 'ArrowLeft') {
            currentIndex = (currentIndex - 1 + galleryImages.length) % galleryImages.length;
            if (lightboxImg) {
                lightboxImg.src = galleryImages[currentIndex];
            }
        } else if (e.key === 'ArrowRight') {
            currentIndex = (currentIndex + 1) % galleryImages.length;
            if (lightboxImg) {
                lightboxImg.src = galleryImages[currentIndex];
            }
        }
    });

    // ============================================
    // Registration Form Bot Protection
    // ============================================
    const registrationForm = document.getElementById('registrationForm');
    let formLoadTime = Date.now();

    if (registrationForm) {
        registrationForm.addEventListener('submit', function (e) {
            // Check if form was submitted too quickly (likely a bot)
            const submitTime = Date.now();
            const timeDiff = submitTime - formLoadTime;

            if (timeDiff < 3000) { // Less than 3 seconds
                e.preventDefault();
                alert('Please take your time to fill out the form properly.');
                return false;
            }

            // Check honeypot fields
            const website = registrationForm.querySelector('[name="website"]');
            const honeypot = registrationForm.querySelector('[name="honeypot_field"]');

            if ((website && website.value) || (honeypot && honeypot.value)) {
                e.preventDefault();
                return false;
            }
        });
    }

    // ============================================
    // Smooth Scroll for Anchor Links
    // ============================================
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            const targetId = this.getAttribute('href');
            if (targetId === '#') return;

            const targetElement = document.querySelector(targetId);
            if (targetElement) {
                e.preventDefault();
                targetElement.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });

    // ============================================
    // Header Scroll Effect
    // ============================================
    const header = document.querySelector('header');
    let lastScrollY = window.scrollY;

    if (header) {
        window.addEventListener('scroll', function () {
            const currentScrollY = window.scrollY;

            if (currentScrollY > 100) {
                header.classList.add('shadow-xl');
            } else {
                header.classList.remove('shadow-xl');
            }

            lastScrollY = currentScrollY;
        });
    }

    // ============================================
    // Form Validation Enhancement
    // ============================================
    const forms = document.querySelectorAll('form');

    forms.forEach(form => {
        const inputs = form.querySelectorAll('input[required], textarea[required], select[required]');

        inputs.forEach(input => {
            input.addEventListener('blur', function () {
                if (!this.value.trim()) {
                    this.classList.add('border-red-500');
                    this.classList.remove('border-gray-300');
                } else {
                    this.classList.remove('border-red-500');
                    this.classList.add('border-gray-300');
                }
            });

            input.addEventListener('input', function () {
                if (this.value.trim()) {
                    this.classList.remove('border-red-500');
                    this.classList.add('border-gray-300');
                }
            });
        });
    });

    // ============================================
    // Lazy Load Images
    // ============================================
    if ('IntersectionObserver' in window) {
        const lazyImages = document.querySelectorAll('img[loading="lazy"]');

        const imageObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    img.style.opacity = '1';
                    observer.unobserve(img);
                }
            });
        });

        lazyImages.forEach(img => {
            img.style.opacity = '0';
            img.style.transition = 'opacity 0.3s ease';
            imageObserver.observe(img);
        });
    }

    // ============================================
    // Console Welcome Message
    // ============================================
    console.log('%c🎓 The Lookeron School', 'color: #1e40af; font-size: 20px; font-weight: bold;');
    console.log('%cLearners Today, Leaders Tomorrow', 'color: #3b82f6; font-size: 14px;');

});

// ============================================
// Dynamic Year for Copyright
// ============================================
document.querySelectorAll('.current-year').forEach(el => {
    el.textContent = new Date().getFullYear();
});

// Hero Slider Initialization 
$(document).ready(function () {
    $('.hero-slider').owlCarousel({
        items: 1,
        loop: true,
        autoplay: true,
        autoplayTimeout: 4000,
        autoplayHoverPause: false,
        nav: true,
        navText: ['<i class="fa-solid fa-chevron-left"></i>', '<i class="fa-solid fa-chevron-right"></i>'],
        dots: true,
        animateOut: 'fadeOut',
        animateIn: 'fadeIn',
        smartSpeed: 2000,
        mouseDrag: true,
        touchDrag: true
    });
});